/*
 * A simple DECnet bridge program
 * (c) 2003, 2005 by Johnny Billquist
 *
 *
 * Original work of Johnny Billquist (bqt@update.uu.se)
 *
 *
 * Subsequent updates:
 *
 *      V0.10-004              FDD + AM + GC                  08-FEB-2026
 *           The DNS refresh code still had problems. Our friend Katolaz
 *           fixed some BSD compatibilty issue with the timeval struct.
 *
 *      V0.10-003              Andrea Milazzo                 26-JUL-2023
 *           Automatic chroot directory in /var/run instead of relying
 *           on passwd entries (e.g. user nobody with a non-existing home
 *           directory by design)
 *
 *      V0.10-002              Gerardo Cacciari               09-JAN-2022
 *           Splitted the [lan] section into specific ones for LAT, MOP, etc.
 *           Added ANF-10 frame filtering (the frame type used by D8EINT.MAC
 *           is 0x6006 but others might have used 0x6016 too)
 *
 *      V0.10-001              Andrea Milazzo                 09-JAN-2022
 *           Long-needed fix to address libpcap compatibility issues. More
 *           robust configuration file handling. General code cleanup.
 *
 *      V0.9-003               Federico Di Dio                02-AUG-2010
 *           Use flags for bridge status.  Don't panic if a hostname can't
 *           be resolved upon startup (thanks to Gerardo and Giuliano for
 *           pointing it out).
 *
 *      V0.9-002               Federico Di Dio                20-DEC-2009
 *           Initial log file support. Fix DNS refresh code. Use standard
 *           EXIT_FAILURE and EXIT_SUCCESS macros. Some things suggested
 *           and/or added by Gerardo Cacciari (mostly comments).
 *
 *      V0.9-001               Federico Di Dio                25-JUN-2009
 *           General cleanup: use tabs for indenting. Do command line
 *           parsing with `getopt'; also changed every switch to lowercase
 *           (-d is now for daemon mode, -g for debug). Reworked DNS refresh
 *           code: now you can force a refresh sending a SIGALRM. Fixed
 *           port number check (do it before `htons'). Really exit on
 *           conflicting daemon and debug options. Group default settings
 *           and put them before the real code.
 *
 *      V0.8-003               Gerardo Cacciari               02-MAR-2009
 *           Reworked some daemonization code: most checks are now done
 *           by parent before forking. Child has just to call setsid() and 
 *           freopen(). This should avoid any startup problems in child. 
 *
 *      V0.8-002               Gerardo Cacciari               01-MAR-2009
 *           Bugfix: pcap_perror() cannot be used if pcap_open_live() fails
 *           because it needs a structure which is set to NULL on error.
 *
 *      V0.8-001               Daniel Kertesz                 28-FEB-2009
 *           New function and command line option to drop root privileges.
 *           If -u is not specified, the program automatically drops to
 *           user nobody. Previous behaviour can be obtained using -u root.
 *
 *      V0.7-004               Gerardo Cacciari               14-AUG-2008
 *           Warning and error handling fully reworked (exits on error but
 *           tries to continue on warning). Old LAT section renamed LAN.
 *           Added InfoServer Ethertype to LAN section. Hard to spot bug
 *           in DATA struct corrected (*data has to be unsigned). 
 *
 *      V0.7-003               Fabio Battaglia                10-JUL-2008
 *           Fixed some glitches with file descriptors when forking
 *
 *      V0.7-002               Federico Di Dio                27-AUG-2007
 *           Fixed minor portability issues, should work on OpenBSD.
 *
 *      V0.7-001               Fabio Battaglia                19-AUG-2007
 *           Added multiple local ethernet support. Now we can listen to and
 *           (if desired) forward between local interfaces. Just add into the
 *           configuration file more than one ethernet line in the [bridge] and
 *           [forward] sections as needed. To forward between any two local
 *           ethernet interfaces enter just one of them into the [forward] sect.
 *
 *      V0.6-003               Gerardo Cacciari               17-AUG-2007
 *           New -b option to force binding to a specified IP address instead
 *           of INADDR_ANY (0.0.0.0). Genericized BSD code using a common ifdef
 *           __BSD__ clause. These changes along with those introduced by
 *           V0.6-1 should cover every routing problem, we hope.
 *
 *      V0.6-002               Gerardo Cacciari               14-AUG-2007
 *           Some minor bug fixes and editing changes. Function dump_data()
 *           splitted into dump_config() and dump_table(). The latter is called
 *           at every DNS lookup cycle when in debug mode. No more special
 *           handling for some signals (SIGHUP and SIGUSR1).
 *
 *      V0.6-001               Gerardo Cacciari               06-AUG-2007
 *           Made some changes coming from Johnny Billquist's bridge v. 2.0
 *           New feature that allows incoming frames from any source UDP port
 *           This new feature is controlled by a new .conf specific section.
 *
 *      V0.5-003               Fabio Battaglia                02-JUN-2007
 *           New genericized daemonization code, some changes to BSD  
 *           conditional includes, some minor cosmetic changes.
 *
 *      V0.5-002               Gerardo Cacciari               04-FEB-2007
 *           New versioning scheme and new change log in the program header
 *           A little banner is displayed upon program startup.
 *
 *      V0.5-001               Gerardo Cacciari               02-FEB-2007
 *           Added selective forward and moved passive settings to a specific
 *           configuration file section (instead of '~' as the first hostname
 *           character in the configuration file). Again some more code
 *           cleanup (mainly debug messages).
 *
 *      V0.4-001               Gerardo Cacciari               24-DEC-2006
 *           Added more debug controls and sanity checks.
 *           Fixed a bug in DNS lookups.
 *
 *      V0.3-002               Fabio Battaglia                21-DEC-2006
 *           Some more debug controls and messages.
 *
 *      V0.3-001               Gerardo Cacciari               21-DEC-2006
 *           Time-controlled DNS lookups (instead of configuration file reload)
 *           and some code cleanup.
 *
 *      V0.2-001               Fabio Battaglia                19-DEC-2006
 *           Command line parsing and daemonization.
 *
 *      V0.1-002               Fabio Battaglia                18-DEC-2006
 *           Configuration file reloadable every N seconds.
 *
 *      V0.1-001               Gerardo Cacciari               18-DEC-2006
 *           Default forwarding disabled (for mesh-structured networks).
 *           First changes over the original bridge from Johnny Billquist.
 *
 *
 * http://decnet.ipv7.net  -  #retrocomputing on irc.azzurra.org
 *
 * Gerardo Cacciari (gerardo.cacciari@gmail.com)
 * Fabio Battaglia (hkzlabnet@gmail.com - http://hkzlab.ipv7.net)
 * Federico Di Dio (eflags@freaknet.org)
 * Daniel Kertesz (daniel@spatof.org)
 * Andrea Milazzo (me@mancausoft.org)
 *
 */


/****************** USER EDITABLE DEFAULT CONFIGURATION  ******************/

#define DEFAULT_PORT 4711
#define DEFAULT_CONFIG "/etc/decnet-bridge.conf"
#define DEFAULT_LOGFILE "/var/log/decnet-bridge.log"
#define DEFAULT_CHROOT "/var/run/decnet-bridge"
#define DEFAULT_USER "nobody"

#define MAX_HOST 16

/*
   Throttling control:
   THROTTLETIME - (uS)
                  If packets come closer in time than this, they are
                  a base for perhaps considering throttling.
   THROTTLEPKT  - (#)
                  The number of packets in sequence that fulfill
                  THROTTLETIME that means throttling will kick in.
   THROTTLEDELAY - (uS)
                  The delay to insert when throttling is active.
*/

#define THROTTLETIME 5000
#define THROTTLEPKT 4
#define THROTTLEDELAY 10

#define PASSIVE_TMO 120000000L

/*************** END OF USER EDITABLE DEFAULT CONFIGURATION ***************/
 
#define BRIDGE_VERSION 0
#define BRIDGE_RELEASE 10
#define BRIDGE_EDITLVL 4

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#   define __BSD__
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <time.h>
#include <sys/ioctl.h>
#ifdef __BSD__
#	include <net/bpf.h>
#else
#	include <pcap-bpf.h>
#endif
#include <fcntl.h>
#include <errno.h>
#include <pcap.h>
#include <netdb.h>
#include <signal.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>

#define CMDLINEMSG "Error parsing command line: "

#define THROTTLEMASK ((1 << THROTTLEPKT) - 1)

#define ETHERTYPE_MOPDL		0x6001
#define ETHERTYPE_MOPRC		0x6002
#define ETHERTYPE_DECNET	0x6003
#define ETHERTYPE_LAT		0x6004
#define ETHERTYPE_ANF10		0x6006
#define ETHERTYPE_LAST		0x8041  /* Infoserver Local Area System Transport */

#define LOCAL		0x0001
#define DONT_RESOLVE	0x0002
#define ANYPORT		0x0004
#define CHECK_LOOP	0x0008
#define FORWARD		0x0010
#define PASSIVE		0x0020

static unsigned int debug = 0;
#define DEBUG if (debug) printf

static unsigned int daemon_mode = 0;
#define TIMESTAMP if (daemon_mode) {\
	char time_buff[30];\
	time_t now;\
	now = time(NULL);\
	strftime(time_buff,30,"%b %d %H:%M:%S", localtime(&now));\
	printf("%s ", time_buff);\
	fflush(stdout);\
}
       	

/*
   This is a very simple and small program for bpf that just
   filters out anything by any protocol that we *know* we're
   not interested in.
*/

struct bpf_insn insns[] = {
	BPF_STMT(BPF_LD + BPF_H + BPF_ABS, 12),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_DECNET, 6, 0),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_LAT, 5, 0),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_MOPRC, 4, 0),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_MOPDL, 3, 0),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_ANF10, 2, 0),
	BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_LAST, 1, 0),
	BPF_STMT(BPF_RET + BPF_K, 0),
	BPF_STMT(BPF_RET + BPF_K, 1518),
};

/* 
   The structures and other global data we keep info in.
   It would perhaps be nice if we could reload this, and
   in case of failure keep the old stuff, but for now we
   don't care that much...

   The data structures we have are the port, which describe
   a source/destination for data. It also holds info about which
   kind of traffic should be forwarded to this site. It is also
   used to filter incoming packets. If we don't send something, we
   don't want it from that side either.
   We have the host table, which is a hash table for all known
   destinations, so that we can optimize the traffic a bit.

   When data arrives, we filter, process and send it out again.

   (original comment from Johnny Billquist bridge)
*/

#define HDRLEN 14		/* Length of the ethernet header */
#define MAXMEM 8		/* Number of old packets we "remember" */

enum pkttyp { DECnet, LAT, MOPRC, MOPDL, ANF10, LAST, MAXTYP };

struct BRIDGE {
	char name[40];
	char host[80];
	in_addr_t addr;
	int port;
	int fd;
	int types[MAXTYP];
	char last[MAXMEM][HDRLEN];
	int lastptr;
	int rcount;
	int tcount;
	int xcount;
	int throttle;
	int throttlecount;
	struct timeval lasttime;
	struct timeval lastrcv;
	int flags;
	pcap_t *pcap;
};

struct DATA {
	int source;
	int type;
	int len;
	struct timeval ts;
	const unsigned char *data;
};

struct HOST {
	struct HOST *next;
	unsigned char mac[6];
	int bridge;
	struct timeval lastseen;
};

#define HOST_HASH 65536

struct HOST *hosts[HOST_HASH];
struct BRIDGE bridge[MAX_HOST];
int bcnt = 0;
int sd;				/* descriptor for udp socket */
volatile char do_resolve = 0;	/* time_interval elapsed flag */

/* Here comes the code... */

/* If the packet come from a known address/port, return the bridge index;
 * return -1 otherwise. */
int lookup(struct sockaddr_in *sa)
{
	int i;

	DEBUG("lookup(): packet from %s port %d ", inet_ntoa(sa->sin_addr), ntohs(sa->sin_port));

	for (i = 0; i < bcnt; i++) {
		if ((bridge[i].addr == sa->sin_addr.s_addr) &&
		    ((bridge[i].flags & ANYPORT) || (bridge[i].port == sa->sin_port))) {
			DEBUG("accepted for bridge %d (%s)\n", i, bridge[i].name);
			return i;
		}
	}
	DEBUG("refused\n");
	return -1;
}


/* Search a bridge by name */
int lookup_bridge(char *newbridge)
{
	int i, bridgelen = strlen(newbridge);
	for (i = 0; i < bcnt; i++)
		if (strcmp(newbridge, bridge[i].name) == 0)
			if  (bridgelen == strlen(bridge[i].name))
				return i;	/* bqt 2.0 */
	return -1;
}


void add_bridge(char *name, char *dst)
{
	char rhost[40];
	int port = 0;
#ifdef __BSD__
	int immed;
#endif
	struct in_addr addr = { 0 };
	char *p;
	struct BRIDGE *newbridge;

	if (bcnt >= MAX_HOST) {
		fprintf(stderr, "%%warning: bridge table full. Not adding %s (%s), continuing\n", name, dst);
		return;
	}
	newbridge = &bridge[bcnt];

	bzero(newbridge, sizeof(struct BRIDGE));
	strncpy(newbridge->name, name, (size_t) 40);

	p = index(dst, ':');
	if (p == NULL) {	/* Assume local descriptor */
		pcap_t *pcap;
		struct bpf_program pgm;
		char ebuf[PCAP_ERRBUF_SIZE];

		newbridge->flags |= LOCAL | DONT_RESOLVE;
		if ((pcap = pcap_open_live(dst, 1522, 1, 1, ebuf)) == NULL) {
			/* pcap_perror(bridge[bcnt].pcap, ebuf); -- V0.8-002 */
			perror("pcap_open_live");
			fprintf(stderr, "%%error: unable to open local device %s, exiting\n", dst);
			exit(EXIT_FAILURE);
		}
		newbridge->pcap = pcap;

		if (pcap_setdirection(pcap, PCAP_D_IN) < 0) {
			pcap_perror(pcap, "setting capture direction");
			fprintf(stderr, "%%warning: unable to set capture direction\n");
			newbridge->flags |= CHECK_LOOP;	/* try workaround */
		}

		pgm.bf_len = sizeof(insns) / sizeof(struct bpf_insn);
		pgm.bf_insns = insns;
		if (pcap_setfilter(pcap, &pgm) == -1) {
			pcap_perror(pcap, "loading filter program");
			fprintf(stderr, "%%error: unable to load filter program, exiting\n");
			exit(EXIT_FAILURE);
		}

		strncpy(newbridge->host, dst, (size_t) 80);
		newbridge->fd = pcap_fileno(pcap);

#ifdef __BSD__
		immed = 1;
		if (ioctl(newbridge->fd, BIOCIMMEDIATE, &immed) == -1) {
			perror("BIOCIMMEDIATE");
			fprintf(stderr, "%%error: ioctl BIOCIMMEDIATE failed, exiting\n");
			exit(EXIT_FAILURE);
		}
		if (ioctl(newbridge->fd, BIOCSHDRCMPLT, &immed) == -1) {
			perror("BIOCSHDRCMPLT");
			fprintf(stderr, "%%error: ioctl BIOCSHDRCMPLT failed, exiting\n");
			exit(EXIT_FAILURE);
		}
#endif
	}
	else {

		*p = ' ';
		sscanf(dst, "%s %d", rhost, &port);
		if (port < 1025 || port > 32767) {
			fprintf(stderr, "%%warning: port number %d for bridge %s is out of range (1025-32767)\n", port, name);
			fprintf(stderr, "-warning: bridge %s definition ignored, continuing\n", name);
			return;
		}
		newbridge->port = htons(port);
		strncpy(newbridge->host, rhost, (size_t) 80);
		newbridge->fd = sd;

		if (inet_aton(rhost, &addr)) {
			newbridge->addr = addr.s_addr;
			newbridge->flags |= DONT_RESOLVE;
		}
		else
			newbridge->addr = INADDR_NONE;
			
	}

	bcnt++;

	DEBUG("Adding bridge %s %s:%d\n", name, inet_ntoa(addr), port);
}


int add_service(char *newbridge, int type, char *name)
{
	int i;
	DEBUG("Adding %s service %s\n", name, newbridge);
	if ((i = lookup_bridge(newbridge)) >= 0) {
		if (bridge[i].types[type]++ > 0) {
			DEBUG("%s service %s added multiple times\n", name, newbridge);
		}
		return 1;
	}
	return 0;
}


int set_passive(char *newbridge)
{
	int i;

	DEBUG("Enabling passive option for bridge %s\n", newbridge);
	if ((i = lookup_bridge(newbridge)) < 0)
		return 0;
	if (bridge[i].flags & PASSIVE)
		DEBUG("Passive option for bridge %s enabled multiple times\n", newbridge);

	bridge[i].flags |= PASSIVE;
	return 1;
}


int set_forward(char *newbridge)
{
	int i;
	
	DEBUG("Enabling forward option for bridge %s\n", newbridge);
	if ((i = lookup_bridge(newbridge)) < 0)
		return 0;
	if (bridge[i].flags & FORWARD)
		DEBUG("Forward option for bridge %s enabled multiple times\n", newbridge);

	bridge[i].flags |= FORWARD;
	return 1;
}


int set_anyport(char *newbridge)
{
	int i;

	DEBUG("Enabling anyport option for bridge %s\n", newbridge);

	if ((i = lookup_bridge(newbridge)) < 0)
		return 0;
	if (bridge[i].flags & ANYPORT)
		DEBUG("Anyport option for bridge %s enabled multiple times\n", newbridge);

	bridge[i].flags |= ANYPORT;
	return 1;
}


void read_conf(const char *config_file)
{
	FILE *f;
	int mode = 0;
	int line, params;
	char buf[80];
	char buf1[40], buf2[40];
/*	int i;*/

	DEBUG("DECnet bridge reading config file\n");

	if ((f = fopen(config_file, "r")) == NULL) {
		perror("fopen");
		fprintf(stderr, "%%error: unable to open configuration file %s, exiting\n", config_file);
		exit(EXIT_FAILURE);
	}

/*	for (i = 0; i < bcnt; i++) {
		if (bridge[i].fd != sd)
			close(bridge[i].fd);
	}
	bcnt = 0;

	for (i = 0; i < HOST_HASH; i++) {
		struct HOST *h, *n;
		h = hosts[i];
		hosts[i] = NULL;
		while (h) {
			n = h->next;
			free(h);
			h = n;
		}
	} only used for dynamic .conf reload */

	line = 0;
	while (!feof(f)) {
		if (fgets(buf, 80, f) == NULL)
			continue;	/* bqt 2.0 */
		buf[strlen(buf) - 1] = 0;
		line++;
		params = sscanf (buf, "%s %s", buf1, buf2);
		if ((strlen(buf1) > 2) && (buf1[0] != '!')) {
			if (buf1[0] == '[') {
				mode = -1;
				if (strcmp(buf1, "[bridge]") == 0)
					mode = 0;
				if (strcmp(buf1, "[decnet]") == 0)
					mode = 1;
				if (strcmp(buf1, "[lan]") == 0)  /* for compatibility with older versions */
					mode = 2;
				if (strcmp(buf1, "[lat]") == 0)
					mode = 3;
				if (strcmp(buf1, "[moprc]") == 0)
					mode = 4;
				if (strcmp(buf1, "[mopdl]") == 0)
					mode = 5;
				if (strcmp(buf1, "[infoserver]") == 0)
					mode = 6;
				if (strcmp(buf1, "[anf10]") == 0 || strcmp(buf1, "[anf-10]") == 0)
					mode = 7;
				if (strcmp(buf1, "[passive]") == 0)
					mode = 8;
				if (strcmp(buf1, "[forward]") == 0)
					mode = 9;
				if (strcmp(buf1, "[anyport]") == 0)
					mode = 10;
				if (mode < 0) {
					fprintf(stderr, "%%warning: bad configuration section \"%s\" at line %d\n", buf, line);
					fprintf(stderr, "-warning: lines up to next section will be ignored, continuing\n");
				}
			}
			else {
				if (mode > 0 && params > 1) {
					fprintf(stderr, "%%warning: bad configuration \"%s\" (too many items) at line %d\n", buf, line);
					continue;
				}
				switch (mode) {
				case 0:
					if (params == 2) {
						add_bridge(buf1, buf2);
					}
					else if (params != -1) { /* ignore blank lines */
						fprintf(stderr, "%%warning: bridge definition \"%s\" at line %d is not correct\n", buf, line);
					}
					break;
				case 1:
					if (!add_service (buf1, DECnet, "DECnet")) {
						fprintf(stderr, "%%warning: DECnet service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 2:
					if (!add_service(buf1, LAT, "LAT")       ||
					    !add_service(buf1, MOPRC, "MOPRC")   ||
					    !add_service(buf1, MOPDL, "MOPDL")   ||
					    !add_service(buf1, ANF10, "ANF-10")  ||
					    !add_service(buf1, LAST, "Infoserver")) {
						fprintf(stderr, "%%warning: LAN service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 3:
					if (!add_service(buf1, LAT, "LAT")) {
						fprintf(stderr, "%%warning: LAT service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 4:
					if (!add_service(buf1, MOPRC, "MOPRC")) {
						fprintf(stderr, "%%warning: MOPRC service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 5:
					if (!add_service(buf1, MOPDL, "MOPDL")) {
						fprintf(stderr, "%%warning: MOPDL service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 6:
					if (!add_service(buf1, ANF10, "ANF-10")) {
						fprintf(stderr, "%%warning: ANF-10 service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 7:
					if (!add_service(buf1, LAST, "Infoserver")) {
						fprintf(stderr, "%%warning: Infoserver service %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 8:
					if (!set_passive(buf1)) {
						fprintf(stderr, "%%warning: passive option %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 9:
					if (!set_forward(buf1)) {
						fprintf(stderr, "%%warning: forward option %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				case 10:
					if (!set_anyport(buf1)) {
						fprintf(stderr, "%%warning: anyport option %s at line %d does not match any bridge definition\n", buf, line);
					}
					break;
				default:
					fprintf(stderr, "%%warning: weird state at line %d\n", line);
				}
			}
		}
	}
	fclose(f);
}


int is_ethertype(struct DATA *d, int type)
{
	return ((d->data[13] == (type & 0xff)) &&
		(d->data[12] == (type >> 8)));
}


int is_decnet(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_DECNET);
}


int is_lan(struct DATA *data)
{
	return (is_ethertype(data, ETHERTYPE_LAT)   ||
		is_ethertype(data, ETHERTYPE_MOPDL) ||
		is_ethertype(data, ETHERTYPE_MOPRC) ||
		is_ethertype(data, ETHERTYPE_LAST));
}


int is_lat(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_LAT);
}


int is_moprc(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_MOPRC);
}


int is_mopdl(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_MOPDL);
}


int is_anf10(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_ANF10);
}


int is_last(struct DATA *data)
{
	return is_ethertype(data, ETHERTYPE_LAST);
}


unsigned long timedelta(struct timeval old)
{
	struct timeval now;
	unsigned long delta;
	gettimeofday(&now, NULL);
	delta = now.tv_sec - old.tv_sec;
	delta *= 1000000;
	delta += (now.tv_usec - old.tv_usec);
	return delta;
}


void throttle(int index)
{
	unsigned long delta;

	delta = timedelta(bridge[index].lasttime);
	bridge[index].throttle <<= 1;
	bridge[index].throttle += (delta < THROTTLETIME ? 1 : 0);

	if ((bridge[index].throttle & THROTTLEMASK) == THROTTLEMASK) {
		bridge[index].throttlecount++;
		usleep(THROTTLEDELAY);
		DEBUG("Throttle active for bridge %d %d(%03o)\n", index, bridge[index].throttlecount, bridge[index].throttle & 255);
	}
	gettimeofday(&bridge[index].lasttime, NULL);
}


int active(int index)
{
	if (!(bridge[index].flags & PASSIVE))
		return 1;
	if (timedelta(bridge[index].lastrcv) < PASSIVE_TMO)
		return 1;
	return 0;
}


/* do the actual sending */
void send_packet(int index, struct DATA *d)
{
	struct sockaddr_in sa;

	if (index == d->source)
		return;		/* Avoid loopback of data. */
	if (bridge[index].types[d->type] == 0)
		return;		/* Avoid sending unwanted frames */

	if ( ((bridge[d->source].flags & LOCAL) == (bridge[index].flags & LOCAL))
	    && !((bridge[d->source].flags | bridge[index].flags) & FORWARD) )
		return;		/* Avoid forward */

	if (active(index)) {
		bridge[index].tcount++;
		throttle(index);
		if (bridge[index].flags & LOCAL) {		/* Local network. */
			write(bridge[index].fd, d->data, d->len);
		}
		else if (bridge[index].addr != INADDR_NONE) {
			sa.sin_family = AF_INET;	/* Remote network. */
			sa.sin_port = bridge[index].port;
			sa.sin_addr.s_addr = bridge[index].addr;
			if (sendto (bridge[index].fd, d->data, d->len, 0, (struct sockaddr *) &sa, sizeof(sa)) == -1) {
				perror("sendto");
				fprintf(stderr, "%%warning: unable to send data to bridge %d (%s), continuing\n",
						 index, bridge[index].name);
			}
		}
		if (bridge[index].flags & CHECK_LOOP) {
			bridge[index].lastptr = (bridge[index].lastptr + 1) & 7;
			memcpy(bridge[index].last[bridge[index].lastptr], d->data, 14);
		}
	
	}
}


void register_source(struct DATA *d)
{
	unsigned short hash;
	struct HOST *h;

	hash = *(unsigned short *) (d->data + 10);
	
	for (h = hosts[hash]; h != NULL; h = h->next) {
		if (memcmp(h->mac, d->data + 6, 6) == 0) {
			h->bridge = d->source;
			if (debug) {
				gettimeofday(&h->lastseen, NULL);
				fprintf(stdout, "Got known host frame from bridge %d ", h->bridge);
				if (is_decnet(d)) {
					fprintf(stdout, "(%d.%d)\n", h->mac[5] >> 2, ((h->mac[5] & 3) << 8) + h->mac[4]);
				}
				else {
			/*		fprintf(stdout, "(LAT, MOP, InfoServer, ANF-10 data)\n"); */
					fprintf(stdout, "(-)\n"); /* less log cluttering */
				}
			}
			return;
		}
	}
	h = malloc(sizeof(struct HOST));
	h->next = hosts[hash];
	memcpy(h->mac, d->data + 6, 6);
	h->bridge = d->source;
	if (debug) {
		gettimeofday(&h->lastseen, NULL);
		fprintf(stdout, "Adding new hash entry for bridge %d ", h->bridge);
		if (is_decnet(d)) {
			fprintf(stdout, "(%d.%d)\n", h->mac[5] >> 2, ((h->mac[5] & 3) << 8) + h->mac[4]);
		}
		else {
	/*		fprintf(stdout, "(LAT, MOP, InfoServer, ANF-10 data)\n"); */
			fprintf(stdout, "(-)\n"); /* less log cluttering */
		}
	}
	hosts[hash] = h;
}


int locate_dest(struct DATA *d)
{
	unsigned short hash;
	struct HOST *h;

	if (d->data[0] & 1)
		return -1;	/* Ethernet multicast */

	hash = *(unsigned short *) (d->data + 4);
	h = hosts[hash];
	while (h) {
		if (memcmp(h->mac, d->data, 6) == 0)
			return h->bridge;
		h = h->next;
	}
	return -1;
}


void process_packet(struct DATA *d)
{
	int dst;
	int i;

	bridge[d->source].rcount++;
	bridge[d->source].lastrcv = d->ts;

	if (bridge[d->source].flags & CHECK_LOOP)
		for (i = 0; i < 8; i++)
			if (memcmp(bridge[d->source].last[i], d->data, 14) == 0)
				return;

	if (d->type == -1)
		return;
	if (bridge[d->source].types[d->type] == 0)
		return;

	bridge[d->source].xcount++;

	register_source(d);
	dst = locate_dest(d);
	if (dst == -1) {
		int i;
		for (i = 0; i < bcnt; i++)
			send_packet(i, d);
	}
	else {
		send_packet(dst, d);
	}
}


void dump_config()
{
	int i;
	struct in_addr addr = { 0 };

	fprintf(stdout, "Host table:\n");
	for (i = 0; i < bcnt; i++) {
		addr.s_addr = bridge[i].addr;
		fprintf(stdout, "%d: %s %s:%d (Rx: %d Tx: %d (Drop rx: %d)) Active: %d Throttle: %d(%03o) Fwd: %d AnyP: %d\n",
			i, bridge[i].name, inet_ntoa(addr),
			ntohs(bridge[i].port), bridge[i].rcount,
			bridge[i].tcount,
			bridge[i].rcount - bridge[i].xcount, active(i),
			bridge[i].throttlecount, bridge[i].throttle & 255,
			(bridge[i].flags & FORWARD)?1:0, (bridge[i].flags & ANYPORT)?1:0);
	}
}


void dump_table()
{
	int i;
	time_t lastseen;

	fprintf(stdout, "Hash of known destinations:\n");
	for (i = 0; i < HOST_HASH; i++) {
		struct HOST *h;
		for (h = hosts[i]; h != NULL; h = h->next) {
			fprintf(stdout, "%02x:%02x:%02x:%02x:%02x:%02x -> %d",
				(unsigned char) h->mac[0],
				(unsigned char) h->mac[1],
				(unsigned char) h->mac[2],
				(unsigned char) h->mac[3],
				(unsigned char) h->mac[4],
				(unsigned char) h->mac[5], h->bridge);
			if ((unsigned char) h->mac[0] == 0xaa
			    && (unsigned char) h->mac[1] == 0x00
			    && (unsigned char) h->mac[2] == 0x04
			    && (unsigned char) h->mac[3] == 0x00) {
				fprintf(stdout, " (%d.%d)", h->mac[5] >> 2,
					((h->mac[5] & 3) << 8) + h->mac[4]);
			}
			lastseen = (time_t) h->lastseen.tv_sec;
						/* time_t != long */
			fprintf(stdout, " last seen %s", ctime(&lastseen));
				/* ctime() provides new line character */
		}
	}
}

void refr_sig_hnd(int signum)
{
	do_resolve = 1;
}

void refresh_ipaddr(unsigned long refresh_interval)
{
	struct hostent *he;
	struct in_addr addr = { 0 };
	int i;
	char new_addr_str[17], old_addr_str[17];
	char *dns_err;

	refresh_interval *= 1000000;
	if (debug)		/* placed here for convenience but has */
		dump_table();	/* nothing in common with DNS lookups  */

	DEBUG("DECnet bridge now looking up hostnames\n");

	for (i = 0; i < bcnt; i++) {
		addr.s_addr = bridge[i].addr;
		strncpy(old_addr_str, (char *) inet_ntoa(addr), 16);
			/* Avoid resolving dotted decimal addresses and */
			/* avoid resolving local bridge definitions */
		if (!(bridge[i].flags & DONT_RESOLVE)) {
			if ((addr.s_addr == INADDR_NONE) ||
			    (timedelta(bridge[i].lastrcv) > (refresh_interval / 2))) {
				DEBUG("Resolving host %s for bridge %s... ", bridge[i].host, bridge[i].name);
				fflush(stdout);
				if ((he = gethostbyname(bridge[i].host)) != NULL) {
					addr.s_addr = *(in_addr_t *) he->h_addr;
					bridge[i].addr = addr.s_addr;
					if (debug) {
						strncpy(new_addr_str, (char *) inet_ntoa(addr), 16);
						DEBUG("OK (old %s - new %s)\n", old_addr_str, new_addr_str);
					}
				}
				else {
					DEBUG("\n");
					switch (h_errno) {
						case HOST_NOT_FOUND:
							dns_err = "HOST_NOT_FOUND";
							break;
						case TRY_AGAIN:
							dns_err = "TRY_AGAIN";
							break;
						case NO_RECOVERY:
							dns_err = "NO_RECOVERY";
							break;
						case NO_ADDRESS:
							dns_err = "NO_ADDRESS";
							break;
						default:
							dns_err = "unknown";
					}
					TIMESTAMP
					fprintf(stderr, "%%warning: unable to resolve `%s' (%s). Using %s\n", bridge[i].host, dns_err, old_addr_str);
					if ((addr.s_addr == INADDR_NONE) && (refresh_interval == 0)) {
						TIMESTAMP
						fprintf(stderr, "%%warning: `%s' will be unreachable.  Send a SIGALRM anytime to try again.\n", bridge[i].name);
					}
					/* herror("gethostbyname"); */
				}
			}
		}
	}
}


void print_usage(char *app_name)
{
	int i = 1;
	char *usage_text[] = {
"Usage: %s [OPTION]...",
""
"  -b ADDR     IP address to bind to (defaulting to 0.0.0.0, i.e. any address)",
"  -c FILE     Non-default configuration file",
"  -d          Run as a daemon",
"  -g          Enable debug",
"  -l LOGFILE  Use LOGFILE instead of " DEFAULT_LOGFILE,
"  -h          Display this help and exit",
"  -p PORT     Non-default port used to receive frames from remote bridges",
"  -r TIME     Enable cyclic DNS lookup for remote host names every TIME",
"              seconds.  A lookup occurs only for bridges which did not",
"              transmit anything in the last (TIME / 2) seconds.",
"              Lookup disabled if not specified",
"  -u USER     Change to user USER after completing privileged operations",
"",
		NULL };


	fprintf(stdout, usage_text[0], app_name);
	fprintf(stdout, "\n");
	while (usage_text[i] != NULL)
		fprintf(stdout, "%s\n", usage_text[i++]);
}


int drop_privs(const char *user_name)
{
	struct passwd *pw;
	struct stat sb;

	if (!(stat(DEFAULT_CHROOT, &sb) == 0 && S_ISDIR(sb.st_mode))) {
		if (mkdir(DEFAULT_CHROOT, 0700) != 0) {
			perror("mkdir");
			return -1;
		}
	}

	pw = getpwnam(user_name);

	if (pw == NULL) {
		perror("getpwnam");
		return -1;
	}

	if (chdir(DEFAULT_CHROOT) < 0) {
		perror("chdir");
		return -1;
	}

	if (chroot(DEFAULT_CHROOT) != 0) {
		perror("chroot");
		return -1;
	}

	if (setgroups(1, &pw->pw_gid) != 0) {
		perror("setgroups");
		return -1;
	}

	if (setgid(pw->pw_gid) != 0) {
		perror("setgid");
		return -1;
	}

	if (setuid(pw->pw_uid) != 0) {
		perror("setuid");
		return -1;
	}

	return 0;
}


int main(int argc, char **argv)
{
	char *config_name = DEFAULT_CONFIG;
	char *user_name = DEFAULT_USER;
	char *logfile = DEFAULT_LOGFILE;
	FILE *logfile_fd;
	struct sockaddr_in sa, rsa;
	int i, hsock, pid, temp;
	long int parse_tmp;
	fd_set full_fds, curr_fds;
	socklen_t ilen;
	struct DATA d;
	unsigned char buf[2048];
	struct sigaction refresh_act;
	struct itimerval refresh;
	char *errmsg = NULL;

	bzero(&refresh_act, sizeof(struct sigaction));
	bzero(&refresh, sizeof(struct itimerval));

	sa.sin_family = AF_INET;
	sa.sin_addr.s_addr = INADDR_ANY;
	sa.sin_port = htons(DEFAULT_PORT);

	fprintf(stdout, "\n  This is DECnet Bridge version V%d.%d-%d\n\n",
			BRIDGE_VERSION, BRIDGE_RELEASE, BRIDGE_EDITLVL);

/* command line parsing and checking */
	opterr = 0;	/* override standard messages */
	while ((temp = getopt (argc, argv, "b:c:dghl:p:r:u:DP:")) != -1)
		switch (temp) {
		case 'b':
			if (!inet_aton(optarg, &sa.sin_addr))
				errmsg = "invalid IP address after -b";
			break;
		case 'c':
			config_name = optarg;
			break;
		case 'g':	/* was 'd' */
			debug = 1;
			break;
		case 'h':
			print_usage(argv[0]);
			exit(EXIT_SUCCESS);
			break;
		case 'l':
			logfile = optarg;
			break;
		case 'r':
			if (!(parse_tmp = strtol(optarg, NULL, 0)))
				errmsg = "value for -r is not numeric";
			else if (parse_tmp < 180)
				errmsg = "value for -r is too small (less than 180 seconds)";
			else {
			       	refresh.it_interval.tv_sec = parse_tmp;
				refresh.it_value.tv_sec = parse_tmp;
			}
			break;
		case 'u':
			user_name = optarg;
			break;
		case 'D':	/* deprecated */
			fprintf(stderr, "%%warning: -D is deprecated and will be removed soon. Please use -d.\n");
		case 'd':	/* fallthrough */
			daemon_mode = 1;
			break;
		case 'P':	/* deprecated */
			fprintf(stderr, "%%warning: -P is deprecated and will be removed soon. Please use -p.\n");
		case 'p':	/* fallthrough */
			if (!(parse_tmp = strtol(optarg, NULL, 0)))
				errmsg = "value for -p is not numeric";
			else if (parse_tmp < 1025 || parse_tmp > 32767)
				errmsg = "value for -p is out of range (1025-32767)";
			else
				sa.sin_port = htons((uint16_t) parse_tmp);
			break;
		case '?':
			switch (optopt) {
			case 'b':
				errmsg = "no IP address given for -b";
				break;
			case 'c':
				errmsg = "no file name given for -c";
				break;
			case 'r':
				errmsg = "no seconds given for -r";
				break;
			case 'u':
				errmsg = "no user name given for -u";
				break;
			case 'p':
			case 'P':
				errmsg = "no port number given for -p";
				break;
			default:
				errmsg = "unknown option";
			}
			break;
		default:
			abort();
		}
	if (optind < argc)
		errmsg = "unknown argument";
	if (errmsg != NULL) {
		fprintf (stderr, CMDLINEMSG "%s\n\n", errmsg);
		print_usage(argv[0]);
		exit(EXIT_FAILURE);
	}

	if (debug && daemon_mode) {
		fprintf(stderr, "%%error: cannot specify both debug (-g) and daemon mode (-d), exiting\n");
		exit(EXIT_FAILURE);
	} 

	if (!daemon_mode) {
		fprintf(stderr, "%%warning: not running in daemon mode: not dropping privileges\n");
	}

	if ((sd = socket(PF_INET, SOCK_DGRAM, 0)) == -1) {
		perror("socket");
		fprintf(stderr, "%%error: unable to open UDP socket, exiting\n");
		exit(EXIT_FAILURE);
	}

	if (bind(sd, (struct sockaddr *) &sa, sizeof(sa)) == -1) {
		perror("bind");
		fprintf(stderr, "%%error: unable to bind to socket, exiting\n");
		exit(EXIT_FAILURE);
	}

	read_conf(config_name);		/* Name resolution must be done
					   before we chroot() to avoid issues
					   with dynamic libraries */
	refresh_ipaddr(refresh.it_interval.tv_sec); 

	if (debug)
		dump_config();

	if (daemon_mode) {
		/* Usually only root is able to open files in /var/log, so
		 * we have to do this before we drop privileges */
		umask(027);
		if ((logfile_fd = fopen(logfile,"a")) == NULL) {
			perror("fopen");
			fprintf(stderr, "%%error: unable to open logfile `%s', exiting\n", logfile);
			exit(EXIT_FAILURE);
		}
		if ((setvbuf(logfile_fd, NULL, _IONBF, BUFSIZ) != 0)) {
			perror("setvbuf");
			fprintf(stderr, "%%warning: could not disable buffering for logfile\n");
		}

		if (drop_privs(user_name) < 0) {
			/*  perror() in drop_privs() */
			fprintf(stderr, "%%error: drop_privs() failed, exiting\n");
			exit(EXIT_FAILURE);
		}

		pid = fork();
		if (pid == -1) {
			perror("fork");	/* fork failed */
			fprintf(stderr, "%%error: unable to fork, exiting\n");
			exit(EXIT_FAILURE);
		}
		else if (pid == 0) {	/* this is the child */
			if (setsid() < 0) {
				perror("setsid");
				fprintf(stderr, "%%error: setsid() in child failed, exiting\n");
				exit(EXIT_FAILURE);
			}
			umask(0);
			freopen("/dev/null", "r", stdin);
			dup2(fileno(logfile_fd), STDOUT_FILENO);
			dup2(fileno(logfile_fd), STDERR_FILENO);
			TIMESTAMP
			fprintf(stdout, "%%success: daemon started\n"); /* this goes to the log */
		}
		else {	/* this is the parent */
			fprintf(stdout, "%%success: daemon started\n"); /* this goes to the parent stdout */ 
			exit(EXIT_SUCCESS);
		}
	}

	/* Asynchronous DNS refresh using alarms
	 *
	 * We setup a signal handler (`refr_sig_hnd') that sets `do_resolve'.
	 * In the main loop we check this flag and eventually do a refresh.
	 * Please note that using the `it_interval' member of 'refresh' we
	 * need to setup the alarm only once.
	 */
	sigemptyset (&refresh_act.sa_mask);
	refresh_act.sa_flags = 0;		/* don't restart syscalls */
	refresh_act.sa_handler = refr_sig_hnd;
	sigaction(SIGALRM, &refresh_act, NULL);

	setitimer(ITIMER_REAL, &refresh, NULL);	/* disabled if refresh is 0 */

	FD_ZERO(&full_fds);
	hsock = 0;
	for (i = 0; i < bcnt; i++) {
		FD_SET(bridge[i].fd, &full_fds);
		if (bridge[i].fd > hsock)
			hsock = bridge[i].fd;
	}
	hsock++;

	for(;;) {
		if (do_resolve) {
			do_resolve = 0;
			refresh_ipaddr(refresh.it_interval.tv_sec);
		}

		curr_fds = full_fds;
		if (select(hsock, &curr_fds, NULL, NULL, NULL) == -1) {
			if (errno == EINTR)	/* got a signal */
				continue;	/* just go on */
			TIMESTAMP
			perror("select");
			TIMESTAMP
			fprintf(stderr, "%%error: main loop select() failed, exiting\n");
			exit(EXIT_FAILURE);
		}

		for (i = 0; i < bcnt; i++) {

			if (! FD_ISSET(bridge[i].fd, &curr_fds))
				continue;

			if (bridge[i].flags & LOCAL) {	/* local */
				struct pcap_pkthdr h;

				d.data = pcap_next(bridge[i].pcap, &h);
				if (d.data == NULL)
					continue;
				d.source = i;
				d.len = h.caplen;
#ifdef __BSD__
				d.ts.tv_sec = (time_t)(h.ts.tv_sec);
				d.ts.tv_usec = (suseconds_t)(h.ts.tv_usec);
#else
				d.ts = h.ts;
#endif
			}
			else {			/* remote */
				ilen = sizeof(rsa);
				if ((d.len = recvfrom(bridge[i].fd, buf, 1522, 0, (struct sockaddr *) &rsa, &ilen)) <= 0)
					continue;

				if ((d.source = lookup(&rsa)) < 0)
					continue;

				gettimeofday(&d.ts, NULL);
				d.data = buf;
			}
			if (is_decnet(&d))
				d.type = DECnet;
			else if (is_lat(&d))
				d.type = LAT;
			else if (is_moprc(&d))
				d.type = MOPRC;
			else if (is_mopdl(&d))
				d.type = MOPDL;
			else if (is_anf10(&d))
				d.type = ANF10;
			else if (is_last(&d))
				d.type = LAST;
			else
				continue;

			process_packet(&d);
			FD_CLR(bridge[i].fd, &curr_fds);
		}
	}
}
